
import sys
# Base4096 decoder included inline
# base4096.py
# Author: Josef Kulovany - ZCHG.org
# Dynamic Base-4096 Encoder/Decoder with Extended Alphabet

import unicodedata
import os

# Generate or load the base4096 character set
def generate_base4096_alphabet(seed):
    seen = set()
    base_chars = []

    # Include seed chars first
    for ch in seed:
        if ch not in seen:
            seen.add(ch)
            base_chars.append(ch)

    # Fill to 4096 with valid Unicode chars
    for codepoint in range(0x20, 0x30000):
        c = chr(codepoint)
        if c not in seen and is_valid_char(c):
            base_chars.append(c)
            seen.add(c)
            if len(base_chars) == 4096:
                break

    if len(base_chars) < 4096:
        raise ValueError("Failed to generate 4096 unique characters.")
    
    return ''.join(base_chars)

# Validity check
def is_valid_char(c):
    try:
        name = unicodedata.name(c)
        return not any(x in name for x in ['CONTROL', 'PRIVATE USE', 'SURROGATE', 'UNASSIGNED', 'TAG'])
    except ValueError:
        return False

SEED = (
    "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
    "!@#$%^&*()-_+=[{]};:',\"<>?/" + ''.join(chr(i) for i in range(0x00, 0x42))
)

def load_frozen_alphabet(filepath="frozen_base4096_alphabet.txt") -> str:
    if not os.path.exists(filepath):
        raise FileNotFoundError(f"Frozen alphabet file not found: {filepath}")

    with open(filepath, "r", encoding="utf-8") as f:
        # Strip whitespace + newlines just in case
        alphabet = f.read().strip().replace("\n", "").replace("\r", "")

    length = len(alphabet)
    unique = len(set(alphabet))

    if length != 4096:
        raise ValueError(f"Frozen alphabet length is {length}, expected 4096 characters.")
    if unique != 4096:
        raise ValueError(f"Frozen alphabet has {unique} unique characters, expected 4096 unique characters.")

    return alphabet

try:
    BASE4096_ALPHABET = load_frozen_alphabet()
except Exception as e:
    # Optional fallback, but warn
    print(f"Warning: Could not load frozen alphabet: {e}")
    print("Falling back to internal seed (not recommended).")
    BASE4096_ALPHABET = generate_base4096_alphabet(SEED)

CHAR_TO_INDEX = {ch: idx for idx, ch in enumerate(BASE4096_ALPHABET)}

# Encoder: bytes → base4096 string
def encode(data: bytes) -> str:
    num = int.from_bytes(data, byteorder='big')
    result = []
    while num > 0:
        num, rem = divmod(num, 4096)
        result.append(BASE4096_ALPHABET[rem])
    return ''.join(reversed(result)) or BASE4096_ALPHABET[0]

# Decoder: base4096 string → bytes
def decode(encoded: str) -> bytes:
    num = 0
    for char in encoded:
        if char not in CHAR_TO_INDEX:
            raise ValueError(f"Invalid character in input: {repr(char)}")
        num = num * 4096 + CHAR_TO_INDEX[char]
    # Determine minimum byte length
    length = (num.bit_length() + 7) // 8
    return num.to_bytes(length, byteorder='big')


hdgl_encoded = '˃ྡ͕ɵΌքξࣼџʅҏ֗ҿऌӯനԟჱՑƔփҠֶߒײ௱ڏίڿېۯੴܠຣݒሒރʵ߁़ׇ߱ࠞƂɴѰًړ˄ਘٛങؠഹࡒഖʴฬЎს̸¾ࡤkصˣิ̄ம˦ิثఁ˩ิআఴˬิණ౿˯ิᅤಲ˲ีȄആ˵ีԍസ˸ีࡁ൰˻ี౮ඥ˾ีၜ෨́ๆąࣟύๆЎएϐๆܭिϓๆૻ९ϖๆ༧তϙๆቬ৮Ϝ็̅ਧϟ็ج੶Ϣ็ইશϥ็ඬଆϨ็ᅥିϫ่ȅஏϮ่Ԏ௬ϱ่ࡂణϴ่౯౨Ϸ่ၝಡϺ้ą೨Ͻ้ЎനЀ้ܭ൞Ѓ้ૻඒІ้༧ැЉ้ቭࣄӌ๖̆ࣿӏ๖حयӒ๖ঈय़ӕ๖තঔӘ๖ᅦৗӛ๗ȆਗӞ๗ԏ੦ӡ๗ࡃણӤ๗౷૭ӧ๗ၞଫӪ๘Ćୱӭ๘ЏூӰ๘ܮఓӳ๘ૼైӶ๘༨ಐӹ๘ቭೆӼ๙̆ഘӿ๙حൊԂ๙ঈඁԅ๙තබԈ๙ᅦฆיຎȇ࣯לຎԐटןຎࡄॏעຎ౸ॿץຎၟহרຏć৾ׯຏАਾײຏܯઓ\u0600ຏ૽ૉ\u0603ຏ༩ଚ؆ຏቮୟ؉ຐ̇ம،ຐخఁ؏ຐউఴؒຐථ౿ؕຐᅧಲؘຑȇആ؛ຑԐസ؟ຑࡄ൰آຑ౸ඥإຑၟ෨بພĈࣟ۫ພБएۮພܰि۱ພ૾९۴ພ༪ত۷ພቯ৮ۺຟ̈ਧ۽ຟد੶܀ຟঊશ܃ຟදଆ܆ຟᅨି܉ຠȈஏ܌ຠԑ௬ܐຠࡅణܓຠ౹౨ܖຠၠಡܙມĈ೨ܜມБനܟມܰ൞ܢມ૾ඒܥມ༪ැܨມተࣄ߽ະ̉ࣿࠀະذयࠃະঋय़ࠆະධঔࠉະᅩৗࠌັȉਗࠏັԒ੦ࠒັࡆણࠕັ౺૭࠘ັၡଫࠛາĉୱࠞາВூࠡາܱఓࠤາ૿ైࠧາ༫ಐࠪາተೆ࠭ຳ̉ഘ࠲ຳذൊ࠵ຳঋඁ࠸ຳධබ࠻ຳᅩฆृໂȊ࣯ॆໂԓटॉໂࡇॏौໂ౻ॿॏໂၢহ॒ໃĊ৾ॕໃГਾक़ໃܲઓज़ໃଁૉफ़ໃ༬ଚॡໃቱୟ।ໄ̊ம१ໄرఁ४ໄঌఴ७ໄන౿॰ໄᅪಲॳໆȊആॶໆԓസॹໆࡇ൰ॼໆ౻ඥॿໆၢ෨ং໖ċࣟઙ໖Дएજ໖ܳिટ໖ଂ९ઢ໖༭তથ໖ቲ৮ન໗̋ਧબ໗ز੶ય໗এશળ໗ඳଆષ໗ᅫି઼໘ȋஏિ໘Ԕ௬ૂ໘ࡈణૅ໘౼౨ૉ໘ၣಡ્໙ċ೨ૡ໙Дന૦໙ܳ൞૩໙ଂඒ૬໙༭ැ૯໙ታࣄఘ༈̌ࣿఛ༈سयఞ༈ঐय़డ༈පঔత༈ᅬৗధ༉Ȍਗఫ༉ԕ੦మ༉ࡉણఱ༉౽૭ఴ༉ၤଫష༊Čୱఽ༊Еூీ༊ܴఓృ༊ଃైే༊༮ಐో༊ታೆౕ་̌ഘౙ་سൊౡ་ঐඁ౦་පබ౩་ᅬฆൢ༜ȍ࣯൧༜Ԗट൪༜ࡊॏ൭༜౾ॿ൰༜ၥহ൳༝č৾൶༝Жਾ൹༝ܵઓർ༝ଅૉൿ༝༯ଚඃ༝ቴୟඇ༞̍மඊ༞شఁඍ༞ওఴඐ༞ඵ౿ඓ༞ᅭಲඖ༟ȍആග༟Ԗസඟ༟ࡊ൰ජ༟౾ඥඥ༟ၥ෨ඨ༬Ď຺ࣟ༬Зएຽ༬ܶिໂ༬ଆ९ໆ༬༰ত໊༬ት৮ໍ༭̎ਧ໒༭ص੶໕༭ঔશ໘༭බଆໝ༭ᅮିༀ༮Ȏஏ༃༮ԗ௬༆༮ࡋణ༉༮౿౨༌༮ၦಡ༏༯Ď೨༒༯Зന༙༯ܶ൞༜༯ଆඒ༟༯༰ැ༢༯ቶࣄရཁ̏ࣿသཁضयအཁকय़ဤཁභঔဧཁᅯৗဪགȏਗိགԘ੦ူགࡌણဳགಀ૭ံགၧଫ္གྷďୱြགྷИூဿགྷܷఓ၂གྷଇై၅གྷ༱ಐ၈གྷቶೆ။ང̏ഘ၎ངضൊၑངಀඥၖངၧ෨ၙདྷԙटᄨདྷࡍॏᄫདྷಁঔᄰདྷቷ৮ᄳན̐ਧᄶནط੶ᄹནখશᄼནමଆᄿནᅰିᅂཔȐமᅆཔطఁᅉཔখఴᅌཔම౿ᅏཔᅰಲᅒཕȐആᅕཕԙസᅘཕࡍ൰ᅛཕಁඥᅞཕၨ෨ᅡརđࣟሤརКएሧརܹिሪརଉ९ርར༳তሰརቸ৮ሳལ̑ਧሶལظ੶ሹལগશሼལඹଆሿལᅱିቂཤȑஏቅཤԚ௬ቈཤࡎణቌཤಂ౨ቑཤၩಡቔཥđ೨ቘཥКനቜཥܹ൞ቡཥଉඒቤཥ༳ැቧཥቹࣅÃྲྀ̒ऀÆྲྀعरÉྲྀঘॠÌྲྀයকÏྲྀᅲড়ÒཷȒਘÕཷԛ੧ØཷࡏતÛཷಃ૮ÞཷၪବáླྀĒ୲äླྀЛெçླྀܺఔêླྀଊొíླྀ༵ಒðླྀቹೇóཹ̒ങöཹعോùཹঘංüཹයභÿཹᅲงǂྈȓࣰǅྈԜठǈྈࡐॐǋྈ಄ঀǎྈၫ়ǑྉēਁǔྉМਿǗྉܻઔǚྉଋોǝྉ༸ଛǠྉቺୠǣྊ̓யǦྊغంǩྊঙవǬྊරಀǯྊᅳಳǲྋȓഇǵྋԜഹǸྋࡐ൱ǻྋ಄ඦǾྋၫ෩ȁྙĔ࣠˄ྙНऐˇྙܼीˊྙଌ॰ˍྙ༹থːྙቻ৯˓ྚ̔ਨ˖ྚػઁ˙ྚচષ˜ྚලଇ˟ྚᅴୀˢྛȔஐ˥ྛԝ௭˨ྛࡑత˫ྛಅ౩ˮྛၬಢ˱ྜĔ೩˴ྜНഩ˷ྜܼൟ˺ྜଌඓ˽ྜ༹ෑ̀ྜቼࣅόྩ̕ऀϏྩؼरϒྩছॠϕྩවকϘྩᅵড়ϛྪȕਘϞྪԞ੧ϡྪࡒતϤྪಆ૮ϧྪၭବϪྫĕ୲ϭྫОெϰྫܽఔϳྫଏొ϶ྫ༼ಒϹྫቼೇϼྫྷ̕ങϿྫྷؼോЂྫྷছංЅྫྷවභЈྫྷᅵงӋྐྵȖࣰӎྐྵԟठӑྐྵࡓॐӔྐྵಇঀӗྐྵၮ়ӚྺĖਁӝྺПਿӠྺܾઔӣྺଐોӦྺ༽ଛөྺችୠӬྻ̖யӯྻؽంӲྻজవӵྻශಀӸྻᅶಳӼྼ̖ങӿྼؽോԂྼজංԅྼශභԈྼᅶงי࿊ȗࣰל࿊Ԡठן࿊ࡔॐע࿊ಈঀץ࿊ၯ়ר࿋ėਁׯ࿋Рਿײ࿋ܿષ\u0602࿋ෂଇ\u0605࿋ᅷୀ؈࿌ȗஐ؋࿌Ԡ௭؎࿌ࡔతؑ࿌ಈ౩ؔ࿌ၯಢؗ࿎ė೩ؚ࿎Рഩ؞࿎ܿൟء࿎ଓඓؤ࿎༾ෑا࿎ቿࣅ۪ဂ̘ऀۭဂؿर۰ဂঞॠ۳ဂၰ়۸ဃȘਘۼဃԡ੧ۿဃࡕત܂ဃಉ૮܅ဃၰବ܈ငĘ୲܋ငСெ\u070fင݀ఔܒငଔొܕင༿ಒܘငቿೇܛစ̘ങܞစؿോܡစঞංܤစසභܧစᅸงߺဒșࣰ߿ဒԢठࠂဒࡖ৯ࠍဓ̙ਨࠐဓـઁࠓဓটષࠖဓහଇ࠙ဓᅹୀࠜနșஐࠟနԢ௭ࠢနࡖతࠥနಊಳࠬပșഇ࠱ပԣࣅॄဢ̚ऀेဢفरॊဢঠॠ्ဢළকॐဢᅺড়॓ဣȚਘॖဣԣ੧ख़ဣࡗતड़ဣಋ૮य़ဣᅺୀॣဤȚஐ०ဤԣ௭३ဤࡗత६ဤಋ౩९ဤၲಢॲဥĚ೩ॵဥУഩॸဥ݂ൟॻဥଖඓॾဥཁෑঁဥኂࣅઘဲ̛ऀછဲقरઞဲডॠડဲෆকતဲᅻড়ધဳțਘફဳԤ੧મဳࡘતલဳಌ૮શဳၳବહဴě୲ાဴФெુဴ݃ఔૄဴଗొૈဴགಒૌဴኂೇૠဵ̛ങૣဵقോ૨ဵডං૫ဵෆභ૮ဵᅻงగ၂Ȝࣰచ၂ԥठఝ၂࡙ॐఠ၂ಎঀణ၂ၴ়ద၃Ĝਁప၃Хਿభ၃݄ઔర၃ଘોళ၃གྷଛశ၃ኃୠహ၄̜யి၄كంూ၄ঢవె၄්ಀొ၄ᅼಳ్၅Ȝഇౘ၅ԥഹౠ၅࡙൱ౣ၅ಎඦ౨၅ၴ෩౫ၒĝ࣠൦ၒЦऐ൩ၒ݅ी൬ၒଙ॰൯ၒངথ൲ၒኄ৯൵ၓ̝ਨ൸ၓلઁൻၓণષൾၓාଇංၓᅽୀආၔȝஐඉၔԦ௭ඌၔ࡚తඏၔಏ౩ඒၔၵಢඕၕĝ೩ඛၕЦഩඞၕ݅ൟඡၕଙඓඤၕངෑටၕኅࣅູၢ̞ऀຼၢمरແၢতॠໄၢැক້ၢኅ৯ໍၣ̞ਨ໒ၣمઁ໕ၣতષ໘ၣැଇໝၣᅾୀༀၤȞஐ༃ၤԧ௭༆ၤ࡛త༉ၤಐ౩༌ၤၶಢ༏ၥĞ೩༒ၥЧഩ༙ၥ݆ൟ༜ၥଚඓ༟ၥཅෑ༢ၥኆࣅရၲ̟ऀသၲنरအၲথॠဤၲၷ়ဩၳğਁာၳШਿုၳ݇ઔဲၳଛોဵၳཆଛးၳኆୠျၴ̟யှၴنం၁ၴথవ၄ၴෑಀ၇ၴᅿಳ၊ၵȟഇ၍ၵԨഹၐၵ࡞൱ၓၵಒඦၖၵၷ෩ၙႂĠ࣠ᄤႂЩऐᄧႂ݈ीᄪႂଜ॰ᄭႂཇথᄰႂኇ৯ᄳႃ̠ਨᄶႃهઁᄹႃদષᄼႃිଇᄿႃᆀୀᅂႄȠஐᅅႄԩ௭ᅈႄࡠతᅋႄಓ౩ᅎႄၸಢᅑႅĠ೩ሢ႒ȡࣰሥ႒Ԫठረ႒ࡡॐራ႒ಔঀሮ႒ၹ়ሱ႓ġਁሴ႓Ъਿሷ႓݉ઔሺ႓ଝોሽ႓ཉଛቀ႓ኈୠቃ႔̡யቆ႔وంቊ႔ধవቍ႔ීಀቒ႔ᆁಳቕ႕ȡഇቚ႕Ԫഹቝ႕ࡡ൱ቢ႕ಔෑቧ႕ኊୖĢୖȢୖ̢ୖЫୖԫୖىୖ݊ୖࡢୖনୖଞୖಕୖුୖཊୖၺୖᆂୖኊୗĢୗȢୗ̢ୗЫୗԫୗىୗ݊ୗࡢୗনୗଞୗಕୗුୗཊୗၺୗᆂୗኊଡ଼Ģଡ଼Ȣଡ଼̢ଡ଼Ыଡ଼ԫଡ଼ىଡ଼݊ଡ଼ࡢଡ଼নଡ଼ଞଡ଼ಕଡ଼ුଡ଼ཊଡ଼ၺଢ଼Ģଢ଼Ȣଢ଼̢ଢ଼Ыଢ଼ԫଢ଼ىଢ଼݊ଢ଼ࡢଢ଼নଢ଼ଞଢ଼ಕଢ଼ුଢ଼ཊଢ଼ၺଢ଼ᆂ୭Ģ୭Ȣ୭̢୭Ы୭ԫ୭ى୭݊୭ࡢ୭ন୭ଞ୭ಕ୭ු୭ཊ୭ၺ୭ᆂ୭ኊ୮Ģ୮Ȣ୮̢୮Ы୮ԫ୮ى୮݊୮ࡢ୮ন୮ଞ୮ಕ୮ු୮ᆂ୯Ģ୯Ȣ୯̢୯Ы୯ԫ୯ى୯݊୯ࡢ୯ন୯ଞ୯ಕஊĢஊȢஊ̢ஊЫஊԫஊىஊ݊ஊࡢஊনஊଞஊಕஊුஊཊஊၺஊᆂஊኊஎĢஎȢஎ̢எЫஎԫஎ݊எࡢஎনஎଞஎಕஎුஎཊஎၺஎᆂஎኊதĢதȢத̢தЫதԫதىத݊தࡢநЫநԫநىந݊நࡢநনநଞநಕநුநཊநၺநᆂநኊனĢனȢன̢னЫனԫனىன݊னࡢனনனଞனಕனුனཊனၺனᆂனኊபĢபȢப̢பЫபԫபىப݊பࡢபনபଞபಕபුபཊபၺபᆂபኊாĢாȢா̢ாЫாԫாىா݊ாࡢாনாଞாಕாුாཊாၺாᆂாኊிĢிȢி̢ிЫிԫிىி݊ிࡢிনிଞிಕிුிཊிၺிᆂிኊீĢீȢீ̢ீЫீԫீىீ݊ீࡢீনீଞீಕீුீཊீၺீᆂீኊுĢுȢு̢ுЫுԫுىு݊ுࡢுনுଞுಕுුுཊுၺுᆂுኊ௨Ģ௨Ȣ௨̢௨Ы௨ԫ௨ى௨݊௨ࡢ௨ন௨ଞ௨ಕ௨ු௨ཊ௨ၺ௨ᆂ௨ኊ௩Ģ௩Ȣ௩̢௩Ы௩ԫ௩ى௩݊௩ࡢ௩ন௩ଞ௩ಕ௩ු௩ཊ௩ၺ௩ᆂ௩ኊ௪Ģ௪Ȣ௪̢௪Ы௪ԫ௪ى௪݊௪ࡢ௪ন௪ଞ௪ಕ௪ු௪ཊ௪ၺ௪ᆂ௪ኊ௫Ģ௫Ȣ௫̢௫Ы௫ԫ௫ى௫݊௫ࡢ௫ন௫ଞ௫ಕ௫ු௫ཊ௫ၺ௫ᆂ௫ኊ௸Ģ௸Ȣ௸̢௸Ы௸ى௸݊௸ࡢ௸ন௸ଞ௸ಕ௸ු௸ཊ௸ኊ௹Ģ௹Ы௹ԫ௹ى௹݊௹ࡢ௹ন௹ଞ௹ಕ௹ු௹ཊ௹ၺ௹ᆂ௹ኊ௺Ģ௺Ȣ௺̢௺Ы௺ԫ௺ى௺݊௺ࡢ௺ন௺ಕ௺ු௺ཊ௺ၺ௺ᆂ௺ኊఀĢఀ̢ఀ݊ఀࡢఀনఀଞఀཊఀၺఀᆂఀኊఎĢఎȢఎ̢ఎЫఎԫఎࡢఎনఎුఎཊఎၺఎᆂఏࡢఏཊఏၺఏኊఐĢఐȢఐ̢ఐЫఐ݊ఐࡢఐনఐଞఐಕఐුఐཊఐၺఐᆂఐኊఒĢఒȢఒ̢ఒЫఒԫఒىఒ݊ఒࡢఒনఒଞఒಕఒුఒཊఒၺఒᆂటȢట̢టЫటىట݊టࡢటনటଞటಕటኊఠĢఠЫఠԫఠىఠ݊ఠࡢఠনఠଞఠಕఠුఠཊఠၺఠᆂఠኊడĢడȢడ̢డЫడԫడىడ݊డࡢడনడಕడුడཊడၺడᆂడኊఢĢఢ̢ఢЫఢىఢ݊ఢনఢଞఢཊఢᆂఢኊరĢరȢర̢రࡢరনరුరཊరၺఱȢఱଞఱಕఱුఱཊఱᆂల݊లࡢలনలଞలಕలුలཊలၺలᆂలኊళĢళȢళ̢ళЫళԫళىళ݊ృȢృ̢ృЫృىృ݊ృࡢృনృଞృಕృුృཊృၺృኊౄĢౄȢౄЫౄԫౄىౄ݊ౄࡢౄনౄଞౄಕౄුౄཊౄၺౄᆂౄኊెĢెȢె̢ెЫెԫెىె݊ెࡢెনెಕెුెཊెၺెᆂెኊేĢే̢ేЫేىే݊ేࡢేনేଞేཊేၺేᆂేኊౢĢౢȢౢ̢ౢЫౢԫౢىౢࡢౢনౢଞౢුౢཊౢၺౣĢ౦Ģ౦Ȣ౦̢౦Ы౦݊౦ࡢ౦ন౦ଞ౦ಕ౦ු౦ཊ౦ၺ౦ᆂ౦ኊ౧Ģ౧Ȣ౧ଞ౧ಕ౧ු౧ཊ౧ၺ౧ᆂ౧ኊ౻Ȣ౻̢౻Ы౻ى౻݊౻ࡢ౻ন౻ଞ౻ಕ౻ු౻ཊ౻ኊ౼Ģ౼Ы౼ԫ౼ى౼݊౼ࡢ౼ন౼ଞ౼ಕ౼ු౼ཊ౼ၺ౼ᆂ౼ኊ౽Ģ౽Ȣ౽̢౽Ы౽ԫ౽ى౽݊౽ࡢ౽ন౽ಕ౽ු౽ཊ౽ၺ౽ᆂ౽ኊ౾Ģ౾̢౾Ы౾ى౾݊౾ࡢ౾ন౾ଞ౾ཊ౾ၺ౾ᆂ౾ኊಋĢಋȢಋ̢ಋЫಋԫಋࡢಋনಋුಋཊಋၺಌىಌ݊ಌࡢಌཊಌၺಌኊಎĢಎȢಎ̢ಎЫಎ݊ಎࡢಎনಎଞಎಕಎුಎཊಎၺಎᆂಎኊಏĢಏȢಏ̢ಏЫಏԫಏىಏ݊ಏࡢಝ̢ಝЫಝىಝ݊ಝࡢಝনಝଞಝಕಝᆂಝኊಞĢಞ̢ಞЫಞԫಞىಞଞಞಕಞཊಞᆂಞኊಟЫಟԫಟনಟଞಟಕಟᆂಟኊಠĢಠȢಠ̢ಠЫಠԫಠىಠ݊ಠࡢಠনಠଞಠᆂಠኊಮĢಮȢಮ̢ಮ݊ಮࡢಮনಮಕಮුಮཊಮၺಯĢಯࡢರ݊ರࡢರনರଞರಕರුರཊರၺರᆂರኊಱĢಱȢಱ̢ಱЫಱԫಱىಱ݊ಱࡢಱনಱଞಱಕುĢುȢು̢ುЫುԫುىು݊ುࡢುনುଞುಕುුುཊುᆂುኊೂĢೂ̢ೂЫೂԫೂىೂ݊ೂࡢೂনೂଞೂಕೂුೂཊೂၺೂᆂೂኊೃĢೃȢೃ̢ೃЫೃԫೃىೃ݊ೃࡢೃনೃಕೃුೃཊೃၺೃᆂೃኊೄĢೄȢೄ̢ೄЫೄԫೄىೄ݊ೄࡢೄনೄଞೄၺೄᆂೄኊೢĢೢȢೢ̢ೢЫೢԫೢ݊ೢࡢೢনೢಕೢුೢཊೢၺೣىೣ݊ೣনೣଞೣಕ೦Ģ೦Ȣ೦̢೦Ы೦݊೦ࡢ೦ন೦ଞ೦ಕ೦ු೦ཊ೦ၺ೦ᆂ೦ኊ೧ࡢ೧ন೧ଞ೧ಕ೧ු೧ཊ೧ၺ೧ᆂ೧ኊംĢംȢം̢ംЫംԫംىം݊ംࡢംনംଞംಕംුംཊംᆂംኊഃĢഃ̢ഃЫഃԫഃىഃ݊ഃࡢഃনഃଞഃಕഃුഃཊഃၺഃᆂഃኊഄĢഄȢഄ̢ഄЫഄԫഄىഄ݊ഄࡢഄনഄಕഄුഄཊഄၺഄᆂഄኊഅĢഅȢഅ̢അЫഅىഅ݊അࡢഅনഅଞഅཊഅၺഅᆂഅኊഔĢഔȢഔ̢ഔЫഔԫഔ݊ഔࡢഔনഔಕഔුഔཊഔၺകىക݊കᆂഖĢഖȢഖ̢ഖЫഖ݊ഖࡢഖনഖଞഖಕഖුഖཊഖၺഖᆂഖኊഗȢഗ̢തĢതȢത̢തЫതԫതىത݊തࡢതনതଞതಕതුതཊതᆂതኊഥĢഥ̢ഥЫഥԫഥىഥ݊ഥࡢഥনഥଞഥಕഥුഥཊഥၺഥᆂഥኊദĢദȢദ̢ദЫദԫദىദ݊ദࡢദনദଞദಕദුദཊദၺദᆂദኊധĢധȢധ̢ധЫധԫധىധ݊ധࡢധনധଞധಕധුധཊധၺധᆂധኊഴĢഴȢഴ̢ഴЫഴԫഴ݊ഴࡢഴনഴಕഴුഴཊഴၺഴᆂഴኊവԫവىവ݊വࡢവনവଞവಕവුവཊവၺവᆂവኊശĢശȢശ̢ശЫശ݊ശࡢശনശଞശಕശුശཊശၺശᆂശኊഷĢഷȢഷ̢ഷЫഷԫഷىഷ݊ഷࡢഷনഷଞഷಕഷුഷཊഷၺഷᆂഷኊൄȢൄ̢ൄЫൄىൄ݊ൄࡢൄনൄଞൄಕൄුൄཊൄၺൄᆂൄኊെĢെȢെ̢െЫെԫെىെ݊െಕെුെཊെၺെᆂെኊേĢേȢേ̢േЫേԫേىേ݊േࡢേনേଞേಕേුേཊേၺേᆂേኊൈĢൈȢൈЫൈԫൈىൈ݊ൈࡢൈনൈଞൈಕൈුൈၺ൚Ģ൚Ȣ൚̢൚Ы൚ԫ൚ى൚݊൚ಕ൚ኊ൛Ģ൛Ȣ൛̢൛Ы൛ԫ൛݊൛ন൛ଞ൛ಕ൛ු൛ཊ൛ၺ൛ᆂ൛ኊ൜݊൜ࡢ൜ন൜ଞ൜ಕ൜ු൜ཊ൜ၺ൜ᆂ൜ኊ൝̢൝Ы൝ԫ൬Ȣ൬̢൬Ы൬ԫ൬ى൬݊൬ࡢ൬ন൬ଞ൬ಕ൬ු൬ཊ൬ၺ൬ᆂ൬ኊ൭Ģ൭Ȣ൭̢൭Ы൭ԫ൭ى൭݊൭ࡢ൭ন൭ଞ൭ಕ൭ු൭ཊ൭ၺ൭ᆂ൭ኊ൮Ģ൮Ȣ൮̢൮Ы൮ԫ൮ى൮݊൮ࡢ൮ন൮ଞ൮ಕ൮ු൮ཊ൮ၺ൮ᆂ൮ኊ൯Ģ൯Ȣ൯̢൯Ы൯ԫ൯ى൯݊൯ࡢ൯ন൯ଞ൯ಕ൯ኊർĢർȢർ̢ർЫർԫർىർ݊ർࡢർনർଞർಕർුർཊർၺർᆂർኊൽĢൽȢൽ̢ൽЫൽԫൽىൽ݊ൽࡢൽনൽଞൽಕൽුඎȢඎ̢ඎԫඎ݊ඎࡢඎনඎଞඎಕඎཊඎၺඎᆂඎኊඏĢඏȢඏ̢ඏЫඏԫඏىඏ݊ඏࡢඏনඏଞඏಕඏුඏཊඏၺඏᆂඏኊඐĢඐȢඐ̢ඐЫඐىඐࡢඐনඐଞඐಕඐුඐཊඐၺඐᆂඐኊඑĢඑȢඑ̢එЫඑԫඑىඑ݊එࡢඑনඑଞඑಕඑුඑཊඑၺඡĢඡȢඡ̢ඡЫඡԫඡ݊ඡনඡଞඡಕඡුඡཊඡၺජĢජȢජ̢ජЫජԫජىජ݊ජࡢජনජଞජཊජၺජᆂජኊනĢනȢන̢නЫනԫනىන݊නࡢනনනଞනಕනුනཊනၺනᆂනኊඳĢඳȢඳ̢ඳԫඳনඳଞඳಕඳුඳཊඳၺඳᆂඳኊපĢපȢප̢පЫපԫපىප݊පࡢපনපଞපಕපුපཊපၺපᆂපኊඵĢඵȢඵ̢ඵЫඵىඵনඵଞඵཊඵၺඵᆂඵኊළĢළȢළ̢ළЫළԫළىළ݊ළࡢළଞළಕළුළཊළၺළᆂළኊෆĢෆȢෆ̢ෆЫෆԫෆىෆ݊ෆࡢෆনෆଞෆಕෆුෆཊෆၺෆᆂෆኊ්Ģ්Ȣ්්̢Ы්ԫ්ى්්݊ࡢ්ন්ଞ්ಕ්ු්ཊාȢා̢ාЫාԫාىා݊ාࡢාনාଞාಕාුාཊාၺාᆂාኊෞĢෞȢෞ̢ෞЫෞԫෞىෞনෞଞෞಕෞුෞཊෞၺෞᆂෞኊෟĢෟȢෟ̢ෟЫෟԫෟىෟ݊ෟࡢෟଞෟಕෟුෟཊෟၺෟᆂෟኊ෦Ģ෦Ȣ෦̢෦Ы෦ԫ෦ى෦݊෦ࡢ෦ন෦ଞ෦ಕ෦ු෦ཊ෦ၺ෦ᆂ෦ኊ෧Ģ෧Ȣ෧̢෧Ы෧ԫ෧ى෧݊෧ࡢ෧ন෧ଞ෧ಕ෧ු෧ཊ෧ᆂ෧ኊขĢขȢข̢ขЫขԫขىข݊ขࡢขনขଞขಕขුขཊขᆂขኊฃĢฃȢฃ̢ฃЫฃԫฃىฃ݊ฃࡢฃপࣀģࣀȣࣀ̣ࣀЬࣀԬࣀيࣀݍࣀࡣࣀপࣀଟࣀಖࣀූࣀཋࣀၻࣀᆃࣀኋࣁģࣁȣࣁ̣ࣁЬࣁԬࣁيࣁݍࣁࡣࣁপࣁଟࣁಖࣁූࣁཋࣁၻࣁᆃࣁኋࣂģࣂȣࣂ̣ࣂЬࣂԬࣂيࣂݍࣂࡣࣂপࣂଟࣂಖࣂූࣂཋࣂၻࣂᆃࣂኋࣃģࣃȣࣃ̣ࣃЬࣃԬࣃيࣃݍࣃࡣࣃপࣃଟࣃಖࣃූࣃཋࣃၻࣃᆃࣃኋࣛģࣛȣ̣ࣛࣛЬࣛԬࣛيࣛݍࣛࡣࣛপࣛଟࣛಖࣛූࣛཋࣛၻࣛᆃࣛኋࣜģࣜȣ̣ࣜࣜЬࣜԬࣜيࣜݍࣜࡣࣜপࣜଟࣜಖࣜූࣜཋࣜၻࣜᆃࣜኋࣝģࣝȣ̣ࣝࣝЬࣝԬࣝيࣝݍࣝࡣࣝপࣝଟࣝಖࣝූࣝཋࣝၻࣝᆃࣝኋࣞģࣞȣ̣ࣞࣞЬࣞԬࣞيࣞݍࣞࡣࣞপࣞଟࣞಖࣞූࣞཋࣞၻࣞᆃࣞኋ࣫ģ࣫ȣ̣࣫࣫Ь࣫Ԭ࣫ي࣫ݍ࣫ࡣ࣫প࣫ଟ࣫ಖ࣫ූ࣫ཋ࣫ၻ࣫ᆃ࣫ኋ࣬ģ࣬ȣ̣࣬࣬Ь࣬Ԭ࣬ي࣬ݍ࣬ࡣ࣬প࣬ଟ࣬ಖ࣬ූ࣬ཋ࣬ၻ࣬ᆃ࣬ኋ࣭ģ࣭ȣ࣭̣࣭Ь࣭Ԭ࣭ي࣭ݍ࣭ࡣ࣭প࣭ଟ࣭ಖ࣭ූ࣭ཋ࣭ၻ࣭ᆃ࣭ኋ࣮ģ࣮ȣ࣮̣࣮Ь࣮Ԭ࣮ي࣮ݍ࣮ࡣ࣮প࣮ଟ࣮ಖ࣮ූ࣮ཋ࣮ၻ࣮ᆃ࣮ኋࣻģࣻȣ̣ࣻࣻЬࣻԬࣻيࣻࡣࣻၻࣼģࣼȣ̣ࣼࣼЬࣼԬࣼيࣼݍࣼࡣࣼপࣼଟࣼಖࣼූࣼཋࣼၻࣼᆃࣼኋࣽģࣽȣ̣ࣽࣽЬࣽԬࣽيࣽݍࣽࡣࣽপࣽଟࣽಖࣽූࣽཋࣽၻࣽᆃࣽኋࣾģࣾȣ̣ࣾࣾЬࣾԬࣾيࣾݍࣾࡣࣾপࣾଟࣾಖࣾූࣾཋࣾၻࣾᆃࣾኋऋģऋȣऋ̣ऋЬऋԬऋيऋݍऋࡣऋপऋଟऋಖऋූऋཋऋၻऋᆃऋኋऌģऌȣऌ̣ऌЬऌԬऌيऌݍऌࡣऌপऌଟऌಖऌූऌཋऌၻऌᆃऌኋऍģऍȣऍ̣ऍЬऍԬऍيऍݍऍࡣऍপऍଟऍಖऍූऍཋऍၻऍᆃऍኋऎģऎȣऎ̣ऎЬऎԬऎيऎݍऎࡣऎপऎଟऎಖऎූऎཋऎၻऎᆃऎኋछģछȣछ̣छЬछԬछيछݍछࡣछপछଟछಖछූछཋछၻछᆃछኋजģजȣज̣जЬजԬजيजݍजࡣजপजଟजಖजූजཋजၻजᆃजኋझģझȣझ̣झЬझԬझيझݍझࡣझপझଟझಖझූझཋझၻझᆃझኋञģञȣञ̣ञЬञԬञيञݍञࡣञপञଟञಖञූञཋञၻञᆃञኋफģफȣफ̣फЬफԬफيफݍफࡣफপफଟफಖफූफཋफၻफᆃफኋबģबȣब̣बЬबԬबيबݍबࡣबপबଟबಖबූबཋबၻबᆃबኋभģभȣभ̣भЬभԬभيभݍभࡣभপभଟभಖभූभཋभၻभᆃभኋमģमȣम̣मЬमԬमيमݍमࡣमপमଟमಖमූमཋमၻमᆃमኋऻģऻȣऻ̣ऻЬऻԬऻيऻݍऻࡣऻপऻଟऻಖऻූऻཋऻၻऻᆃऻኋ़ģ़ȣ़़̣Ь़Ԭ़ي़ݍ़ࡣ़প़ଟ़ಖ़ූ़ཋ़ၻ़ᆃ़ኋऽģऽȣऽ̣ऽЬऽԬऽيऽݍऽࡣऽপऽଟऽಖऽූऽཋऽၻऽᆃऽኋाģाȣा̣ाЬाԬाيाݍाࡣाপाଟाಖाූाཋाၻाᆃाኋोģोȣो̣ोЬोԬोيोݍोࡣोপोଟोಖोූोཋोၻोᆃोኋौģौȣौ̣ौЬौԬौيौݍौࡣौপौଟौಖौූौཋौၻौᆃौኋ्ģ्ȣ््̣Ь्Ԭ्ي्ݍ्ࡣ्প्ଟ्ಖ्ූ्ཋ्ၻ्ᆃ्ኋॎģॎȣॎ̣ॎЬॎԬॎيॎݍॎࡣॎপॎଟॎಖॎූॎཋॎၻॎᆃॎኋज़ģज़ȣज़̣ज़Ьज़Ԭज़يज़ݍज़ࡣज़পज़ಖज़ූज़ཋज़ၻड़ģड़ȣड़̣ड़Ьड़Ԭड़يड़ݍड़পड़ಖड़ූड़ཋड़ၻढ़ģढ़ȣढ़̣ढ़Ьढ़Ԭढ़يढ़ݍढ़ࡣढ़পढ़ଟढ़ಖढ़ූढ़ཋढ़ၻढ़ᆃढ़ኋफ़ģफ़ȣफ़̣फ़Ьफ़Ԭफ़يफ़ݍफ़ࡣफ़পफ़ଟफ़ಖफ़ූफ़ཋफ़ၻफ़ᆃफ़ኋ५ģ५ȣ५̣५Ь५Ԭ५ي५ݍ५ࡣ५প५ಖ५ූ५ཋ५ၻ६ģ६ȣ६̣६Ь६Ԭ६ي६ݍ६ࡣ६প६ଟ६ಖ६ූ६ཋ६ၻ६ᆃ६ኋ७ģ७ȣ७̣७Ь७Ԭ७ي७ݍ७ࡣ७প७ଟ७ಖ७ಹय़ඐၔ̢ಝ݂ોࠗ႔༲ਪǉेၻऎଞ൚ቹೈǃरॄႤૄ๘ၺขপभඵଭûਧൺႲІႲױႲâѩǃࣆǅୁǃจǉਙôখùତي५ЬऋဌၓቭષԃႥܑ࿌ݍࣽଟञ݊ஊූ़ىඑၺ௺Ģ୭ᅪ࣯ॄႥඅႲܣႲభဤቲఁၘໆ̢ඏၺഔԫ౦ܾुåටþऀÊຑၺඡಕீԘஏઝႲమၔ੍ႤໆႲయၳଟ࣬ȔൌöඁӜྐྵࡋকӭႤᅖႤቍႲӥႥǺႤຽႤ࠲Ⴅဵၥ̢ൄনඐ࡚ୁåഺâતð่ූࣂၺ෦Ģஎුජᅽູ৾ႤનႥ་ྻԫൄ݊ಝ౷ഺè୳ÿથǈৰäೇࠓႥӤ่ኅॐफ़ནၯरఛႲöມኂ૯ìଇˡ၃ཊംᆃ५ԩઌၶઓॖႥౌ๙ĢீුೂࡢಏȢಌ༮ഺíౡଞഷኊఏࡣाيࣻМඓЃႥ༊ႲॆႤ໒Ⴄ༃Ⴒ܌ႤæႥेႥǮ༜ຬႥॶႲ׳Ⴄ༠ྚଟ५Ԭࣂಖफၺ୭Ģஊܹോ\u070fႥఘႥျཹᅾൠåଫأၵᅸથý੨çఓ൮ဵ࡞࣡èથåਧࠨႤǎၓ̢௩݊ෆܹਙèૌǄ౪ùకǃਾశ༯ුఠО୳äൊϔႲϏྩಕరු෧གਪùથùਂĀँǉෑϬໂቷಡӠๆࡅోä౪ñ൲úൠëࣱǄశìଜǅࣆĀ৯؉ႥॣႤۮႲԊႲӓဂ಄ંãୠϵໂࡌමǇഺîആઙၵၺநԕॱüೆۭႥ൷ႥÞဢଇ৮థႤᄬྋᆂ௸ಕ౽Ģడ౿࣠ॕဃࡢഔĢಎࢸ0©é´ۿᄖ˄ࡤଁծӏҒԍ,ڿຜ̗´ݣےޢઃݒ׃Ɇבڿπ߀ֵʴ৬ɴ`ݣծڟưɉʧޣҺޒծٿຨۯƤۀѺۏં۟׃ݳʫݓѫٿᄆɉʷݎµۀӀݮªۀʧä͢ڏƵڼѻϞڗݳ৬ɴ$ߑҳݣɼɆఄޢַɆൗܰመٿᄛɄྐ،Βҽᄑޒݶɴ&ࠒᄃ݀੭ɆʣޢջΌvέ*ݒκگ׃̶Ұڟሆۀɢ߰ઉ۫*ࠃҰݒҰګ$ܱªٿʧެଡ଼ۿညݣʶɉֿۿκگֵڀҬèઁݲሔޮ±ޜଡ଼ɴ)ڿᄇޒƶڻ±ގ®ݢƦɉҳڻ¤ڀΰΌvέ¥ۯƴٿοۀɢޢׅèҰۋ©ڿᄇޒƶھሄڀΰΌvΰሃܱªٿʧޮशڿֵ˕ଡ଼ɄWɉΰڿႺНµۀѳ˒୵ɄWڏƵھህۯƴޞzɇഹâ୵ɄWɴ_ݒηߐҰɉΰڿѫڟतޓ͢ۏઇޣѕɄWɈۘގ¥۫«ݎµڿֵϫ୵ɄWɄWɈੰɈγɈᄑޮ«ݎµڿֿϫ୵ɄWɄWɄWɄµڿֿ̨Ʀګद۫৬ɄWɄWɄWɄWڏƵھህۯƴޞᄃݳ§ݒѳڟ࣬â୵ɄWɴ(ۿຢɉҺɅѻϞډ߰ઉ۫¸ٿຟګ<ݒ੭ݢҰɈγڀʵäWɄ¨ݣɢڟሆۀ±ۿᄖɈંɉʣݒߟ˅ºͥiɅº͵mͅfϫ୵ɄWɄWɈ͢Н¥۰ɪڟሆۀ±ۿᄖ˒୵ɄWɄWɈੰɈ͢ݒሖɈંɉΰڿႺٿᄆɈઈ٠ۊܰ੮ٟγڀɪڛਥäWɄWɄWɄWɈʣޢְڟतޓͰڀ²ڿᄆˈͫäWɄWɄWɄWɉΰڿაٿүˈͫäWɄWɄWɄWɈੰɈປݎथڀΰٟγڀʵ˔zНqͅਡϫ୵ɄWɄWɄWɄWɄWڐʧٿೃäWɄ«ۋ®ڿჂڏƵھህۯƴޞਅЍqͅਡϫ୵ɄWɄWɉʣ܀ΰɇۊֶܱҢʴݣɪɦۊۿປګ¶ݞ©ڿᄇޒƶڻqͅਡɉֿ܀Ʒڻ¥ۯƴٿοۀʵ̤ɫäWɄAɄWɉʧ߁׃ݎdʴᄌݢંˈʣޢְڟतޓͫâஂɇۊܰ੮܀ӄɈγڿζèҰۋ«ޡመٿຟڮህۯƴˈͫϫ୵ɄW߁ʻϫ୵ɄWɄWɈᄃ݀ծН·ݒ੭ݢҰگƶٻᄐٿ࿘ˈͫäWɄWɄWޒׅߑʰɈᄑޮ£ݓਓ߾«ݎ°ٿ࿘ɈۘގºɈંɇഁѿᇰ֖ʑԍޘɄߊղੑѠҐɇ֤ҝޘɄߍ֦ʔՀޭ֕յ̄d֥უփΓҿყҍޘɄߎџޓػ৬ɄWɈבڟׁޮ>ٿອڽ׃ޒሔϫ୵ɄWɄWɉʧ߁׃ݎ(ٿຫڹଡ଼ւ֖ҍzɄࣂɄWɄɲ͕ɵΌքξࣼџʅҏ֗ҿऌӯനԟჱՑƔփҠֶߒײ௱ڏίڿېۯੴܠຣݒሒރʵ߁़ׇ߱ࠞɌɄWɄɣэͦʗჀˤ̗࣬ᇍР൝ـྫྷϭޘثɾЮᇑɤhɄޓ̨ఄۿჂڟवˈਔɈۘގ«Ɉંɉʣݒߟ˅ºͅiɅºΌɫ˒உâ௴ڿډܰሃڮለޒሜڿᄁٿຨۯƤۀѳۏ੶ۀ£߀ऀɨۛݣ௵ݑሄڀΰΌvΰሃܱªٿʧޮᄖࠃѭ˔jЭµ߁ɼäWɄ«ۋ°ݣѫݣͰݲƶ۫ᄇࠂઈ߁ͪۏ੶ۀ£߀࣬ϫ୵ɄWɄWɉʣ܀ΰɆےܰ֟ݣґݣֿڭ׃ޒሔˈڋҲʱֿࠢɈƮݲतڏׅɈےܰծݒሖɈۘߐᄆϭ½ۏ੶ۀ£߀ीɤ৬äWɄ¹܀ҳɈሒڿჂۏ੶ۀ£߀࣯ɄʴɤสڿᄅݢҴݒީɩׅۋྩɤਅڀ͢یଡ଼ɄWɄWɄZɇοޒઅɉߢ܀Ұޣ£ڟծ˴°ۀߦۿᄇޞ¬ߑοɈંɈάޢՖɄWɄWɄ£ܱªٿʧޮzɈڗޒֲګ̩࣬οޒઅ˄ਙޒׁܰƥڻࣥدႼ̄Yɤਙޒׁܰƥڻࣥذɤ̄Yɤ৬äWɄ®ڿᄉ߀ࣣН®ڿჂٿຨۯƤۀѴäWɄ·ݒઆߐծН®ڿჂޢׅˈƮݲतڏׅ˔৬äWɄ«ۋ®ڿᄉ߀ࣣɕྐΌvήଡ଼ɄWɄWɄ´ٿઈڻ>ٿອڽ׃ޒሔˈڋҲʱֿࠢɈƮݲतڏׅɈປݒ߮۫«ޞ½ֿܰ۠ҳࡑสۀळڿοڿѫΌvέ¥ۯƴٿοۀʵ̤ɫäWɄ«ۋ·ݒઆߐծɕྐΌvήଡ଼ɄWɄWɄ´ٿઈڻ>ٿອڽ׃ޒሔˈڋҲʱֿࠢɈƮݲतڏׅɈतޞ½ߐᄋރֶࡑ·ݒઆߐծڟतޒƥ߀׃ޞสۀळڿοڿѫΌvέ·ݒઆߐծڟतޒƥ߀׃ޞႼ˒ଡ଼ɄWɉʧ߁׃ݎ£ܱªٿʧެଡ଼߁ʻϫ୵ɄWѯƕҞѻϞۈџຄӏƄҠѫН®ݢƦٟۛݣ௵ݑሃܱªٿʧޮ࣬èבڟׁޮ*ࠂΰݳҴݢႺڀ͢ڼଡ଼ɄWɄ͢Ճ¶ۿሐٿสۏƮܰʣڟആɈʷޮ¹ڀʰäWɄ²ޒંޮऩɧߛޒᄋݒަɆκߐບɈᄑޮ®ݢƦɈۛݣ௵ݎ£ܱªٿʧޯ୵࠴זɤ৬ɄWɉ´ۿᄖ˄ʈٿຢۿᄉɈʣڟ೬߀ᇂۿᄖۀʰٿสޢֶګeݒሖɉʧڟሏֿ݀گֵ˔Ⴜ˒୵ɄWѯƕҞѻϞۈџຄӏƄҠѫН©ڿᄇޒƶھሄڀΰΌvΰሃܱªٿʧޮखҟ֕˒ଡ଼ѿऄճᇶՁᇫԯҐכzɉി۬୵ۿӃɈۘގ«ڰ࣯ɈγɈંɈֿߐ࿘ޒƶڻअѠΏΌvΰᇣԐ-џʇ֓ઔâஂɆֿڟሆۀɼɈʻ߀ׄɐˈਇ¤ڀΰΌvέµ߁ʫݒݶگַɈֿڟሆڻधڀҬϭ¤ࠓҰޞਅ̕Ⴚޣҽϫ୵ɄWݓ־Ʌྐۿᄖ̨ۛݢ࿒ڐઉۀͪگƶٻสڐઉڿሔگ׃Нߜۿޓ˒୵ɄWޒׄߐຬɅྐ؞ླྀɄWɉߢۿປɈᄗܼ!ɅwäWɄWɄWݓ־̄´ڿྐН¦܀ۖݢѳݓ־̄qͅਡ˒୵ɄWɄWɉʧޣֽޮᄃݳ§ݒѳѯƕҞѻϞۈџຄӏƄҠҦޒ־ػ৬ɄWɉʧ߁׃ݎdʴᄌݢંˉʧߠ׃ޢֵˉʧޣֽޮਔɈሔɆʃււͅਡٝƎՐऄѯ֥؛.âஂɆҰڟሆۀɼɈʣޢւͅਡɉοޒંۛĤबɢڐઉۀ͌گַɈҰڟሆڻनݒκگֵϭµ߁ɫɄྯɈʻ߀ׄϫ୵ɄWݓ־Ʌྐ͂୵ɄWۏሔɈγڀɢۿႺڿᄅݢҰڬଡ଼ɄWɄWɄ«ۋ¥ۯƴɈᄑޮ«ݎ^ӏƔٞҚٝੀҏ֩ϫ୵ɄWɄWɄWɄ´ٿઈڻ>ٿອڽ׃ޒሔˈڋӡᄘٿຟګ¥ۯƴٿοۀɢۿႺۿᄒߑ҅ɉൔۀ´ˈγڀɫࡑɫäWɄWɄWݓ־Ʌྐݓ־Ʉ୵ΌvέhɆΒѠʡ֕ሁӟღҠञڟतޑླྀɄWɄ͢ґׅۀʯۿᄇɈဃݒઁߐྐڐઉڻ®ڿᄉ߀ࣂɄWɈປݒ߮۫zɄऱߐྟڏઉٟປݒ߮۫࣬Ʉ೬νਅ̴ᇂϋ୵ɄWޒׅߑʰɈᄗܼᄖݡሄࠓҰޞयڿᄉ߀࣯Ɉʻ߀׀ޒҰޏྗڏੱʴ৬'
hdgl_bytes = decode(hdgl_encoded)

# Extract alphabet
a_len = int.from_bytes(hdgl_bytes[:2],'big')
alphabet = hdgl_bytes[2:2+a_len].decode('utf-8')

# Extract fingerprint
f_len = int.from_bytes(hdgl_bytes[2+a_len:4+a_len],'big')
fingerprint = hdgl_bytes[4+a_len:4+a_len+f_len].decode('utf-8')

# Extract provisioner
p_len = int.from_bytes(hdgl_bytes[4+a_len+f_len:8+a_len+f_len],'big')
provisioner = hdgl_bytes[8+a_len+f_len:8+a_len+f_len+p_len].decode('utf-8')

# Execute provisioner
exec(provisioner)
